#include "DS1Wire.h"
#include "mbed.h"
#include <stdint.h>

// Timing delay for 1-wire serial standard option
enum DELAY { A = 6, B = 64, C = 60, D = 10, E = 9, F = 55, G = 0, H = 480, I = 70, J = 410 };

   
int Reset(DigitalInOut& pin) {
    pin.output();
    pin = 0;        // drive bus low
    wait_us(H);     //复位脉冲：主机通过拉低单总线至少480μs产生Tx复位脉冲，然后由主机释放总线，并进入Rx接收模式;
    pin.input();    // release bus
    wait_us(I);     //应答脉冲：主机释放总线时，会产生一由低电平跳变为高电平的上升沿，单总线器件检测到该上升沿后，
                    //延时15～60μs，接着单总线器件通过拉低总线60～240μs来产生应答脉冲;
    uint32_t result = pin;  // read bus value
    wait_us(J);
    return result;
}

//写信号：所有的读、写时序至少需要60μs，且每两个独立的时序之间至少需要1μs的恢复时间。写时序均始于主机拉低总线，
//若主机在拉低总线15μs之内释放总线，表示单总线器件写1；若主机拉低总线后能保持至少60μs的低电平，则表示向单总线器件写0;
void WriteBit(DigitalInOut& pin, uint32_t bit) {
    pin.output();
    if (bit) {
        pin = 0;        // drive bus low
        wait_us(A);        // delay A
        pin.input();      // release bus
        wait_us(B);        // delay B
    } else {
        pin = 0;    // drive bus low
        wait_us(C);    // delay C
        pin.input();  // release bus
        wait_us(D);    // delay D
    }
}

//读信号：读时序也始于主机拉低总线，在主机发出读时序之后，单总线器件开始在总线上发送0或1。若单总线器件发送1，则总线保持高电平，若发送0，则拉低总线。
//由单总线器件发送数据后可保持15μs有效时间，因此，主机在读时序期间必须释放总线，且须在15μs的采样总线状态，以便接收从机发送的数据。
uint32_t ReadBit(DigitalInOut& pin) {
    uint32_t bit_value;
    pin.output();
    pin = 0;        // drive bus low
    wait_us(A);        // delay A
    pin.input();      // release bus
    wait_us(E);        // delay E
    bit_value  = pin;    // master sample bus
    wait_us(F);
    return bit_value;
}

void WriteByte(DigitalInOut& pin, uint32_t byte) {
    for (uint32_t bit = 0; bit < 8; ++bit) {
        WriteBit(pin, byte & 0x01); // lsb to msb
        byte >>= 1;    // right shift by 1-bit
    }
}

uint32_t ReadByte(DigitalInOut& pin) {
    uint32_t byte = 0;
    for (uint32_t bit = 0; bit < 8; ++bit) {
        byte |= (ReadBit(pin) << bit);    // Reads lsb to msb
    }
    return byte;
}
